<?php
// ipcodetv.com/generate-sitemap.php
// Run once to (re)build /sitemap.xml (and optional gzip copy)

declare(strict_types=1);
error_reporting(E_ALL & ~E_NOTICE);

// ---- config ---------------------------------------------------------------
$BASE     = 'https://ipcodetv.com';
$POST_DIR = __DIR__ . '/Blog/posts';
$HOME     = __DIR__ . '/index.html'; // change to index.php if needed
$WRITE_GZ = true;                    // also write sitemap.xml.gz

// ---- helpers --------------------------------------------------------------
function abs_url(string $base, string $maybe): string {
  // if already absolute
  if (preg_match('~^https?://~i', $maybe)) return $maybe;
  return rtrim($base, '/') . '/' . ltrim($maybe, '/');
}

function encode_path_component(string $s): string {
  // keep / but encode any unsafe path chars
  return implode('/', array_map('rawurlencode', explode('/', $s)));
}

function safe_loc(string $url): string {
  // do not encode scheme/host; encode path/query fragments safely
  $parts = parse_url($url);
  if (!$parts || empty($parts['scheme']) || empty($parts['host'])) return $url;

  $scheme = $parts['scheme'] . '://';
  $host   = $parts['host'];
  $port   = isset($parts['port']) ? (':' . $parts['port']) : '';
  $path   = isset($parts['path']) ? encode_path_component($parts['path']) : '/';
  $query  = isset($parts['query']) ? '?' . $parts['query'] : '';
  $frag   = isset($parts['fragment']) ? '#' . $parts['fragment'] : '';

  return $scheme . $host . $port . $path . $query . $frag;
}

function lastmod_from_file(?string $file): string {
  if ($file && is_file($file)) return gmdate('c', filemtime($file));
  return gmdate('c');
}

// ---- collect URLs ---------------------------------------------------------
$urls = [];

// Home
$urls[] = [
  'loc'        => safe_loc($BASE . '/'),
  'lastmod'    => lastmod_from_file($HOME),
  'changefreq' => 'daily',
  'priority'   => '1.0',
  'images'     => []
];

// Blog index (only if it exists & returns 200 normally in your setup)
$urls[] = [
  'loc'        => safe_loc($BASE . '/Blog/posts/'),
  'lastmod'    => gmdate('c'),
  'changefreq' => 'daily',
  'priority'   => '0.9',
  'images'     => []
];

// Blog posts
if (is_dir($POST_DIR)) {
  foreach (glob($POST_DIR . '/*.html') as $file) {
    $basename = basename($file); // e.g. blog-xxx.html
    $locRaw   = $BASE . '/Blog/posts/' . $basename;
    $loc      = safe_loc($locRaw);
    $last     = lastmod_from_file($file);
    $images   = [];

    $html = @file_get_contents($file);
    if ($html !== false) {
      // prefer og:image
      if (preg_match('~<meta\s+property=["\']og:image["\']\s+content=["\']([^"\']+)~i', $html, $m)) {
        $img = abs_url($BASE, html_entity_decode($m[1], ENT_QUOTES | ENT_HTML5));
        $images[] = safe_loc($img);
      } elseif (preg_match('~<img\s+[^>]*src=["\']([^"\']+)["\']~i', $html, $m)) {
        $img = abs_url($BASE, html_entity_decode($m[1], ENT_QUOTES | ENT_HTML5));
        $images[] = safe_loc($img);
      }
    }

    $urls[] = [
      'loc'        => $loc,
      'lastmod'    => $last,
      'changefreq' => 'weekly',
      'priority'   => '0.8',
      'images'     => $images
    ];
  }
}

// sort newest first (optional)
usort($urls, function($a, $b){
  return strcmp($b['lastmod'], $a['lastmod']);
});

// ---- build XML ------------------------------------------------------------
$dom = new DOMDocument('1.0', 'UTF-8');
$dom->formatOutput = true;

$urlset = $dom->createElement('urlset');
$urlset->setAttribute('xmlns', 'http://www.sitemaps.org/schemas/sitemap/0.9');
$urlset->setAttribute('xmlns:image', 'http://www.google.com/schemas/sitemap-image/1.1');
$dom->appendChild($urlset);

foreach ($urls as $u) {
  $url = $dom->createElement('url');

  $loc = $dom->createElement('loc');
  $loc->appendChild($dom->createTextNode($u['loc']));
  $url->appendChild($loc);

  $lastmod = $dom->createElement('lastmod');
  $lastmod->appendChild($dom->createTextNode($u['lastmod']));
  $url->appendChild($lastmod);

  $url->appendChild($dom->createElement('changefreq', $u['changefreq']));
  $url->appendChild($dom->createElement('priority',   $u['priority']));

  foreach ($u['images'] as $img) {
    $imgNode = $dom->createElement('image:image');
    $imgLoc  = $dom->createElement('image:loc');
    $imgLoc->appendChild($dom->createTextNode($img));
    $imgNode->appendChild($imgLoc);
    $url->appendChild($imgNode);
  }

  $urlset->appendChild($url);
}

$xml = $dom->saveXML();

// ---- write files ----------------------------------------------------------
file_put_contents(__DIR__ . '/sitemap.xml', $xml);
if ($WRITE_GZ) {
  file_put_contents(__DIR__ . '/sitemap.xml.gz', gzencode($xml, 9));
}

// Also respond with the XML so you can open this script in a browser
header('Content-Type: application/xml; charset=UTF-8');
echo $xml;
